<?php
defined('ABSPATH') or die();

class WPTech_Audit
{
    public static function run_audit()
    {
        return array(
            'security' => self::check_security(),
            'performance' => self::check_performance(),
            'compatibility' => self::check_compatibility(),
            'stability' => self::check_stability(),
        );
    }

    // --- BEZPIECZEŃSTWO ---
    private static function check_security()
    {
        $items = [];

        // 1. Prefiks bazy danych
        if (self::has_default_prefix()) {
            $items[] = [
                'status' => 'danger',
                'title' => 'Prefiks bazy danych – domyślny',
                'details' => '
                    <p><strong>Dlaczego to źle?</strong><br>Domyślny prefiks <code>wp_</code> ułatwia ataki SQL Injection, bo struktura tabel jest przewidywalna.</p>
                    <p><strong>Jak naprawić?</strong></p>
                    <ol>
                        <li>Zrób kopię zapasową.</li>
                        <li>W pliku <code>wp-config.php</code> zmień <code>$table_prefix</code>.</li>
                        <li>Przemianuj tabele w bazie danych.</li>
                    </ol>'
            ];
        } else {
            $items[] = [
                'status' => 'ok',
                'title' => 'Prefiks bazy danych – zmieniony',
                'details' => '<p><strong>Prefiks bazy danych został zmieniony.</strong><br>To zwiększa odporność na ataki SQL Injection.</p>'
            ];
        }

        // 2. XML-RPC
        $xmlrpc_status = self::check_xmlrpc_status();
        if ($xmlrpc_status['enabled']) {
            $items[] = [
                'status' => 'warning',
                'title' => 'XML-RPC – włączone',
                'details' => '
                    <p><strong>Dlaczego XML-RPC może być niebezpieczne?</strong><br>XML-RPC umożliwia zdalne zarządzanie WordPress, ale może być wykorzystywane do ataków brute-force i DDoS.</p>
                    <p><strong>Jak wyłączyć XML-RPC?</strong></p>
                    <p><strong>Opcja 1: Przez wtyczkę</strong></p>
                    <ol>
                        <li>Zainstaluj wtyczkę <strong>Disable XML-RPC</strong>.</li>
                        <li>Aktywuj wtyczkę.</li>
                    </ol>
                    <p><strong>Opcja 2: Przez .htaccess</strong></p>
                    <pre># Blokuj XML-RPC
&lt;Files xmlrpc.php&gt;
    Order Allow,Deny
    Deny from all
&lt;/Files&gt;</pre>
                    <p><strong>Opcja 3: Przez kod w functions.php</strong></p>
                    <pre>add_filter(\'xmlrpc_enabled\', \'__return_false\');</pre>'
            ];
        } else {
            $items[] = [
                'status' => 'ok',
                'title' => 'XML-RPC – wyłączone',
                'details' => '<p><strong>XML-RPC jest prawidłowo wyłączone.</strong><br>To zmniejsza ryzyko ataków brute-force i DDoS na endpoint xmlrpc.php.</p>'
            ];
        }

        // 3. Problematyczne wtyczki
        $problematic_plugins = self::get_problematic_plugins();
        $items[] = empty($problematic_plugins) ? [
            'status' => 'ok',
            'title' => 'Wtyczki – bez zagrożeń',
            'details' => '<p><strong>Nie znaleziono nieużywanych ani podejrzanych wtyczek.</strong><br>To zmniejsza ryzyko wykorzystania luk bezpieczeństwa.</p>'
        ] : [
            'status' => 'warning',
            'title' => 'Podejrzane wtyczki – ' . count($problematic_plugins),
            'details' => '
                <p><strong>Dlaczego to ważne?</strong><br>Stare lub nieaktywne wtyczki mogą zawierać luki bezpieczeństwa.</p>
                <p><strong>Co robić?</strong></p>
                <ol>
                    <li>Usuń wtyczki, których nie używasz.</li>
                    <li>Zaktualizuj te, które mają brak wersji lub są nieaktywne od lat.</li>
                </ol>
                <ul><li>' . implode('</li><li>', $problematic_plugins) . '</li></ul>'
        ];

        // 4. Uprawnienia plików
        $bad_permissions = self::get_insecure_permissions();
        $items[] = empty($bad_permissions) ? [
            'status' => 'ok',
            'title' => 'Uprawnienia – poprawne',
            'details' => '<p><strong>Pliki i katalogi mają bezpieczne uprawnienia.</strong><br>To chroni je przed nieautoryzowaną modyfikacją.</p>'
        ] : [
            'status' => 'warning',
            'title' => 'Niepoprawne uprawnienia – ' . count($bad_permissions),
            'details' => '
                <p><strong>Dlaczego to ważne?</strong><br>Nieprawidłowe uprawnienia umożliwiają modyfikację plików przez atakującego.</p>
                <p><strong>Zalecane uprawnienia:</strong></p>
                <ul>
                    <li>Pliki: <code>644</code></li>
                    <li>Katalogi: <code>755</code></li>
                    <li><code>wp-config.php</code>: <code>600</code> lub <code>640</code></li>
                </ul>
                <p><strong>Do poprawy:</strong></p>
                <ul><li>' . implode('</li><li>', array_map('esc_html', $bad_permissions)) . '</li></ul>'
        ];

        // 5. Konto "admin"
        if (self::admin_user_exists()) {
            $items[] = [
                'status' => 'danger',
                'title' => 'Użytkownik "admin" istnieje',
                'details' => '
                    <p><strong>Dlaczego to źle?</strong><br>„admin" to najczęstszy cel ataków brute-force.</p>
                    <p><strong>Jak naprawić?</strong></p>
                    <ol>
                        <li>Utwórz nowe konto administratora.</li>
                        <li>Przypisz mu uprawnienia.</li>
                        <li>Usuń konto "admin".</li>
                    </ol>'
            ];
        } else {
            $items[] = [
                'status' => 'ok',
                'title' => 'Brak użytkownika "admin"',
                'details' => '<p><strong>Konto "admin" nie istnieje.</strong><br>To zmniejsza ryzyko ataków brute-force.</p>'
            ];
        }

        // 6. WP_DEBUG + WP_DEBUG_DISPLAY
        $is_debug_risky = self::is_debug_risky();
        $items[] = $is_debug_risky ? [
            'status' => 'warning',
            'title' => 'WP_DEBUG jest włączony i widoczny',
            'details' => '
                <p><strong>Dlaczego to niebezpieczne?</strong><br>Widoczne błędy PHP mogą ujawniać dane systemowe lub strukturę serwera.</p>
                <p><strong>Jak naprawić?</strong></p>
                <p>W pliku <code>wp-config.php</code> ustaw:</p>
                <pre>define(\'WP_DEBUG\', false);
        define(\'WP_DEBUG_DISPLAY\', false);</pre>'
        ] : [
            'status' => 'ok',
            'title' => 'WP_DEBUG – wyłączony lub niewidoczny',
            'details' => '<p><strong>Ustawienia debugowania nie zagrażają bezpieczeństwu.</strong><br>Debugowanie nie jest aktywne lub nie jest widoczne dla użytkowników.</p>'
        ];

        // 7. Pliki PHP w uploads i dostępność HTTP
        $php_in_uploads = self::get_accessible_php_files_in_uploads();
        $items[] = !empty($php_in_uploads) ? [
            'status' => 'danger',
            'title' => 'Pliki PHP w katalogu uploads dostępne publicznie',
            'details' => '
                <p><strong>Dlaczego to źle?</strong><br>Katalog <code>uploads</code> nie powinien zawierać plików PHP dostępnych przez przeglądarkę – mogą one zostać wykonane przez atakującego.</p>
                <p><strong>Jak naprawić?</strong></p>
                <ol>
                    <li>Usuń pliki PHP z <code>wp-content/uploads</code>.</li>
                    <li>Zablokuj ich uruchamianie przez .htaccess:</li>
                </ol>
                <pre>&lt;FilesMatch "\.php$"&gt;
                    Deny from all
                    &lt;/FilesMatch&gt;</pre>
                <p><strong>Wykryte pliki:</strong></p>
                <ul><li>' . implode('</li><li>', array_map('esc_html', $php_in_uploads)) . '</li></ul>'
        ] : [
            'status' => 'ok',
            'title' => 'Pliki PHP w uploads – brak publicznego dostępu',
            'details' => '<p><strong>Katalog <code>uploads</code> jest bezpieczny.</strong><br>Nie znaleziono żadnych plików PHP dostępnych przez przeglądarkę.</p>'
        ];

        // 8. Directory Browsing
        $directory_browsing = self::check_directory_browsing();
        if ($directory_browsing['enabled']) {
            $items[] = [
                'status' => 'warning',
                'title' => 'Directory Browsing – włączone',
                'details' => '
                    <p><strong>Dlaczego to niebezpieczne?</strong><br>Włączone przeglądanie katalogów pozwala atakującym na poznanie struktury plików i katalogów Twojej strony.</p>
                    <p><strong>Jak wyłączyć Directory Browsing?</strong></p>
                    <p><strong>Dodaj do pliku .htaccess:</strong></p>
                    <pre># Wyłącz przeglądanie katalogów
Options -Indexes</pre>
                    <p><strong>Lub dodaj do wp-config.php:</strong></p>
                    <pre>// Wyłącz directory browsing
ini_set(\'auto_detect_line_endings\', false);</pre>
                    <p><strong>Wykryte dostępne katalogi:</strong></p>
                    <ul><li>' . implode('</li><li>', array_map('esc_html', $directory_browsing['accessible_dirs'])) . '</li></ul>'
            ];
        } else {
            $items[] = [
                'status' => 'ok',
                'title' => 'Directory Browsing – wyłączone',
                'details' => '<p><strong>Przeglądanie katalogów jest prawidłowo wyłączone.</strong><br>Atakujący nie mogą poznać struktury Twoich plików i katalogów.</p>'
            ];
        }

        // 9. File Editing w Admin Panel
        if (self::is_file_editing_enabled()) {
            $items[] = [
                'status' => 'warning',
                'title' => 'Edycja plików w admin – włączona',
                'details' => '
                    <p><strong>Dlaczego to ryzykowne?</strong><br>Edytor plików w panelu admin może być wykorzystany przez atakującego po przejęciu konta do modyfikacji plików strony.</p>
                    <p><strong>Jak wyłączyć edycję plików?</strong></p>
                    <p><strong>Dodaj do wp-config.php (przed linią "/* That\'s all, stop editing! */":</strong></p>
                    <pre>// Wyłącz edycję plików w admin panelu
define(\'DISALLOW_FILE_EDIT\', true);</pre>
                    <p><strong>Po dodaniu tej linii:</strong></p>
                    <ul>
                        <li>Zniknie edytor motywów i wtyczek z panelu admin.</li>
                        <li>Pliki będzie można edytować tylko przez FTP/SSH.</li>
                    </ul>'
            ];
        } else {
            $items[] = [
                'status' => 'ok',
                'title' => 'Edycja plików w admin – wyłączona',
                'details' => '<p><strong>Edycja plików przez panel admin jest wyłączona.</strong><br>To zabezpiecza przed modyfikacją plików strony w przypadku przejęcia konta administratora.</p>'
            ];
        }

        // 10. SSL i HTTPS
        $ssl = self::get_ssl_status();
        if ($ssl['is_https'] && $ssl['url_forced_https']) {
            $items[] = [
                'status' => 'ok',
                'title' => 'SSL i HTTPS – aktywne',
                'details' => '<p><strong>Strona korzysta z bezpiecznego połączenia.</strong><br>
        Certyfikat SSL jest aktywny, a połączenia HTTPS są wymuszane – to chroni dane przesyłane między użytkownikiem a serwerem.</p>'
            ];
        } elseif ($ssl['is_https']) {
            $items[] = [
                'status' => 'warning',
                'title' => 'SSL aktywny, ale brak wymuszenia HTTPS',
                'details' => '
                    <p><strong>Dlaczego to problem?</strong><br>Użytkownicy mogą nadal trafić na niezaszyfrowaną wersję strony (HTTP).</p>
                    <p><strong>Jak naprawić?</strong></p>
                    <ol>
                        <li>W WordPressie ustaw <code>https://</code> jako adres strony.</li>
                        <li>Dodaj przekierowanie w pliku <code>.htaccess</code>:</li>
                    </ol>
                    <pre>
                        RewriteEngine On
                        RewriteCond %{HTTPS} off
                        RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
                    </pre>'
            ];
        } else {
            $items[] = [
                'status' => 'danger',
                'title' => 'Brak SSL',
                'details' => '
                    <p><strong>Dlaczego SSL jest ważny?</strong><br>Bez niego dane przesyłane między użytkownikiem a serwerem są niezaszyfrowane.</p>
                    <p><strong>Jak naprawić?</strong></p>
                    <ol>
                        <li>Zainstaluj certyfikat SSL (np. Let\'s Encrypt).</li>
                        <li>W WordPressie ustaw adres na <code>https://</code>.</li>
                        <li>Dodaj przekierowanie w <code>.htaccess</code>.</li>
                    </ol>'
            ];
        }

        return $items;
    }
        
    // --- WYDAJNOŚĆ ---
    private static function check_performance()
    {
        $items = [];

        // 1. TTFB
        $ttfb = self::get_page_load_time_frontend();
        if ($ttfb['ttfb_ms'] !== null && $ttfb['ttfb_ms'] <= 500) {
            $items[] = [
                'status' => 'ok',
                'title' => 'Czas odpowiedzi serwera (TTFB) – bardzo dobry',
                'details' => '
                    <p><strong>TTFB wynosi ' . intval($ttfb['ttfb_ms']) . ' ms</strong> – to bardzo dobry wynik.</p>
                    <p>Świadczy o szybkim serwerze lub skutecznie działającym cache.</p>'
            ];
        } elseif ($ttfb['ttfb_ms'] !== null) {
            $items[] = [
                'status' => 'warning',
                'title' => 'Czas odpowiedzi serwera (TTFB) – zbyt wysoki',
                'details' => '
                    <p><strong>TTFB wynosi ' . intval($ttfb['ttfb_ms']) . ' ms</strong> – powyżej zalecanego progu 500 ms.</p>
                    <p><strong>Dlaczego to ważne?</strong></p>
                    <ul>
                        <li>Wydłuża ładowanie strony.</li>
                        <li>Może oznaczać brak cache lub wolny hosting.</li>
                    </ul>
                    <p><strong>Jak poprawić?</strong></p>
                    <ol>
                        <li>Włącz cache HTML i obiektowy.</li>
                        <li>Optymalizuj zapytania do bazy danych.</li>
                        <li>Rozważ lepszy hosting.</li>
                    </ol>'
            ];
        }

        // 2. Rozmiar strony
        $page_size = self::get_page_size();
        if ($page_size <= 2) {
            $items[] = [
                'status' => 'ok',
                'title' => 'Rozmiar strony – w normie',
                'details' => '<p><strong>Rozmiar strony to ' . number_format($page_size, 2) . ' MB</strong> – mieści się w zalecanym limicie 2 MB.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'Rozmiar strony – zbyt duży',
                'details' => '
                    <p><strong>Rozmiar strony to ' . number_format($page_size, 2) . ' MB</strong> – zalecany maks. to 2 MB.</p>
                    <p><strong>Dlaczego to źle?</strong></p>
                    <ul>
                        <li>Wolniejsze ładowanie, zwłaszcza na urządzeniach mobilnych.</li>
                    </ul>
                    <p><strong>Jak to poprawić?</strong></p>
                    <ol>
                        <li>Kompresuj obrazy (np. Imagify, ShortPixel).</li>
                        <li>Minifikuj i scalaj CSS/JS.</li>
                        <li>Używaj lazy-load dla obrazów i iframe.</li>
                    </ol>'
            ];
        }

        // 3. Redis / Memcached
        if (self::is_object_cache_properly_configured()) {
            $items[] = [
                'status' => 'ok',
                'title' => 'Cache obiektowy (Redis/Memcached) – aktywny',
                'details' => '<p><strong>Cache obiektowy działa prawidłowo.</strong><br>Wykryto Redis lub Memcached oraz aktywny plik <code>object-cache.php</code>.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'Cache obiektowy (Redis/Memcached) – nieaktywny',
                'details' => '
                    <p><strong>Nie wykryto aktywnego cache obiektowego.</strong></p>
                    <p><strong>Dlaczego to ważne?</strong></p>
                    <ul>
                        <li>Brak cache oznacza wolniejsze zapytania do bazy danych.</li>
                    </ul>
                    <p><strong>Jak to włączyć?</strong></p>
                    <ol>
                        <li>Zainstaluj Redis lub Memcached na serwerze.</li>
                        <li>Skorzystaj z wtyczki <code>Redis Object Cache</code> lub <code>W3 Total Cache</code>.</li>
                    </ol>'
            ];
        }

        // 4. Cache HTML
        if (self::is_html_cache_enabled()) {
            $items[] = [
                'status' => 'ok',
                'title' => 'Cache strony (HTML) – aktywny',
                'details' => '<p><strong>Cache HTML jest aktywny.</strong><br>Wykryto nagłówki cache lub aktywną wtyczkę buforującą stronę.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'Cache strony (HTML) – brak lub niewykryty',
                'details' => '
                    <p><strong>Nie wykryto mechanizmu buforowania HTML.</strong></p>
                    <p><strong>Jak poprawić?</strong></p>
                    <ol>
                        <li>Zainstaluj i skonfiguruj wtyczkę cache: <strong>LiteSpeed Cache</strong>, <strong>WP Rocket</strong> lub <strong>W3 Total Cache</strong>.</li>
                        <li>Upewnij się, że cache działa dla użytkowników niezalogowanych.</li>
                    </ol>'
            ];
        }

        // 5. CDN
        $cdn = self::detect_cdn();
        if ($cdn) {
            $items[] = [
                'status' => 'ok',
                'title' => 'CDN – aktywny (' . esc_html($cdn) . ')',
                'details' => '<p><strong>Strona korzysta z CDN.</strong><br>Wykryto aktywny system CDN (<strong>' . esc_html($cdn) . '</strong>) – zasoby są szybciej serwowane użytkownikom.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'CDN – nieaktywny',
                'details' => '
                    <p><strong>Nie wykryto aktywnego CDN.</strong></p>
                    <p><strong>Dlaczego warto korzystać z CDN?</strong></p>
                    <ul>
                        <li>Szybsze ładowanie zasobów globalnie.</li>
                        <li>Redukcja obciążenia serwera.</li>
                    </ul>
                    <p><strong>Jak włączyć CDN?</strong></p>
                    <ol>
                        <li>Zarejestruj się w Cloudflare lub BunnyCDN.</li>
                        <li>Skonfiguruj proxy dla swojej domeny.</li>
                    </ol>'
            ];
        }

        // 6. Limit pamięci PHP
        $mem_limit = self::get_php_memory_limit();
        if ($mem_limit >= 256) {
            $items[] = [
                'status' => 'ok',
                'title' => 'Limit pamięci PHP – odpowiedni',
                'details' => '<p><strong>Limit pamięci PHP jest wystarczający.</strong><br>Obecny limit to ' . esc_html($mem_limit) . ' MB.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'Limit pamięci PHP – zbyt niski',
                'details' => '
                    <p><strong>Limit pamięci PHP to tylko ' . esc_html($mem_limit) . ' MB.</strong></p>
                    <p><strong>Jak zwiększyć?</strong></p>
                    <pre>define(\'WP_MEMORY_LIMIT\', \'256M\');</pre>'
            ];
        }

        // 7. Max execution time
        $max_exec = self::get_max_execution_time();
        if ($max_exec >= 60) {
            $items[] = [
                'status' => 'ok',
                'title' => 'Czas wykonywania PHP – odpowiedni',
                'details' => '<p><strong>Limit czasu wykonywania skryptów PHP jest wystarczający.</strong><br>Obecny limit to ' . intval($max_exec) . ' sekund.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'Czas wykonywania PHP – zbyt niski',
                'details' => '
                    <p><strong>Obecny limit czasu to tylko ' . intval($max_exec) . ' sekund.</strong></p>
                    <p><strong>Jak zwiększyć?</strong></p>
                    <pre>max_execution_time = 120</pre>'
            ];
        }

        return $items;
    }

    // --- AKTUALNOŚĆ ---
    private static function check_compatibility()
    {
        global $wp_version, $wpdb;
        $items = [];

        // === PHP VERSION ===
        $php_version = self::get_php_version();
        if (version_compare($php_version, '8.0.0', '>=')) {
            $items[] = [
                'status' => 'ok',
                'title' => "PHP: {$php_version}",
                'details' => '
                    <p><strong>Wersja PHP jest aktualna.</strong><br>
                    Twoja instalacja korzysta z nowoczesnej i wspieranej wersji PHP.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => "PHP: {$php_version} – przestarzała",
                'details' => '
                    <p><strong>Dlaczego warto zaktualizować PHP?</strong></p>
                    <ul>
                        <li>Starsze wersje mogą zawierać luki bezpieczeństwa.</li>
                        <li>Nowe wtyczki mogą nie być z nimi zgodne.</li>
                    </ul>
                    <p><strong>Jak zaktualizować?</strong></p>
                    <ol>
                        <li>Wejdź do panelu hostingu (np. cPanel, Plesk).</li>
                        <li>Wybierz wersję minimum PHP 8.0.</li>
                        <li>Sprawdź działanie strony po zmianie.</li>
                    </ol>'
            ];
        }

        // === WORDPRESS VERSION ===
        $latest_wp = self::get_latest_wp_version();
        if (version_compare($wp_version, $latest_wp, '>=')) {
            $items[] = [
                'status' => 'ok',
                'title' => "WordPress: {$wp_version}",
                'details' => '
                    <p><strong>WordPress jest aktualny.</strong><br>
                    Twoja instalacja jest zgodna z najnowszymi standardami bezpieczeństwa i kompatybilności.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => "WordPress: {$wp_version} – dostępna {$latest_wp}",
                'details' => '
                    <p><strong>Dlaczego warto aktualizować WordPress?</strong></p>
                    <ul>
                        <li>Poprawki bezpieczeństwa i wydajności.</li>
                        <li>Nowe funkcje i zgodność z wtyczkami.</li>
                    </ul>
                    <p><strong>Jak zaktualizować?</strong></p>
                    <ol>
                        <li>Przejdź do <strong>Kokpit → Aktualizacje</strong>.</li>
                        <li>Wykonaj kopię zapasową.</li>
                        <li>Kliknij <strong>Zaktualizuj teraz</strong>.</li>
                    </ol>'
            ];
        }

        // === BAZA DANYCH ===
        $db_version = $wpdb->db_version();
        $engine = self::get_database_engine_type();
        $latest_db = $engine === 'MariaDB' ? self::get_latest_mariadb_version() : self::get_latest_mysql_version();

        if (version_compare($db_version, $latest_db, '>=')) {
            $items[] = [
                'status' => 'ok',
                'title' => "{$engine}: {$db_version}",
                'details' => "
                    <p><strong>Wersja {$engine} jest aktualna.</strong><br>
                    Silnik bazy danych nie wymaga aktualizacji – działa zgodnie z zaleceniami.</p>"
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => "{$engine}: {$db_version} – dostępna {$latest_db}",
                'details' => "
                    <p><strong>Dlaczego warto aktualizować {$engine}?</strong></p>
                    <ul>
                        <li>Lepsze bezpieczeństwo i wydajność bazy danych.</li>
                    </ul>
                    <p><strong>Jak zaktualizować?</strong></p>
                    <ol>
                        <li>Skontaktuj się z obsługą hostingu.</li>
                        <li>Upewnij się, że strona i wtyczki są kompatybilne.</li>
                    </ol>"
            ];
        }

        // === WTYCZKI ===
        $outdated_plugins = self::get_outdated_plugins_list();
        if (empty($outdated_plugins)) {
            $items[] = [
                'status' => 'ok',
                'title' => 'Wtyczki – aktualne',
                'details' => '<p><strong>Wszystkie wtyczki są zaktualizowane.</strong><br>
                Nie ma potrzeby podejmowania żadnych działań.</p>'
            ];
        } else {
            $list = '<ul><li>' . implode('</li><li>', array_map('esc_html', $outdated_plugins)) . '</li></ul>';
            $items[] = [
                'status' => 'warning',
                'title' => 'Nieaktualne wtyczki – ' . count($outdated_plugins),
                'details' => '
                    <p><strong>Dlaczego warto aktualizować wtyczki?</strong></p>
                    <ul>
                        <li>Nowe wersje zawierają poprawki bezpieczeństwa.</li>
                        <li>Poprawiają stabilność działania strony.</li>
                    </ul>
                    <p><strong>Jak zaktualizować?</strong></p>
                    <ol>
                        <li>Wejdź w <strong>Wtyczki → Zainstalowane</strong>.</li>
                        <li>Wykonaj kopię zapasową.</li>
                        <li>Zaktualizuj wskazane wtyczki.</li>
                    </ol>
                    <p><strong>Wtyczki wymagające aktualizacji:</strong></p>' . $list
            ];
        }

        // === AUTOMATYCZNE AKTUALIZACJE ===
        $is_auto = self::is_auto_update_enabled();
        if ($is_auto) {
            $items[] = [
                'status' => 'ok',
                'title' => 'Automatyczne aktualizacje – włączone',
                'details' => '
                    <p><strong>Automatyczne aktualizacje rdzenia są aktywne.</strong><br>
                    Twoja strona otrzymuje automatyczne poprawki bezpieczeństwa bez ingerencji użytkownika.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'Automatyczne aktualizacje – wyłączone',
                'details' => '
                    <p><strong>Dlaczego warto je włączyć?</strong></p>
                    <ul>
                        <li>Poprawki bezpieczeństwa są instalowane automatycznie.</li>
                    </ul>
                    <p><strong>Jak włączyć?</strong></p>
                    <ol>
                        <li>Otwórz <code>wp-config.php</code>.</li>
                        <li>Dodaj linię:<br><code>define(\'WP_AUTO_UPDATE_CORE\', true);</code></li>
                    </ol>'
            ];
        }

        return $items;
    }

    // --- STABILNOŚĆ ---
    private static function check_stability()
    {
        $items = [];

        // 1. CRON WordPress
        if (self::has_cron_jobs()) {
            $items[] = [
                'status' => 'ok',
                'title' => 'WordPress CRON – działa poprawnie',
                'details' => '<p><strong>CRON działa prawidłowo.</strong><br> Zaplanowane zadania są aktywne i wykonywane automatycznie.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'WordPress CRON – brak zaplanowanych zadań',
                'details' => '
                    <p><strong>Dlaczego CRON jest ważny?</strong></p>
                    <ul>
                        <li>Odpowiada za automatyczne zadania (publikacje, czyszczenie bazy, aktualizacje).</li>
                    </ul>
                    <p><strong>Jak naprawić?</strong></p>
                    <ol>
                        <li>Sprawdź, czy w <code>wp-config.php</code> NIE ma: <code>define(\'DISABLE_WP_CRON\', true);</code></li>
                        <li>Wejdź w <strong>Narzędzia > Zdrowie witryny</strong>.</li>
                        <li>Skonfiguruj CRON systemowy jeśli to konieczne:<br><code>wget -q -O - ' . site_url('/wp-cron.php?doing_wp_cron') . '</code></li>
                    </ol>'
            ];
        }

        // 2. REST API
        if (self::is_rest_api_available()) {
            $items[] = [
                'status' => 'ok',
                'title' => 'REST API – aktywne',
                'details' => '<p><strong>REST API działa poprawnie.</strong><br> Wymagane dla Gutenberga i integracji z zewnętrznymi systemami.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'REST API – niedostępne',
                'details' => '
                    <p><strong>Dlaczego REST API jest ważne?</strong></p>
                    <ul>
                        <li>Obsługuje edytor blokowy i zewnętrzne integracje.</li>
                    </ul>
                    <p><strong>Jak naprawić?</strong></p>
                    <ol>
                        <li>Sprawdź, czy żadna wtyczka (np. Disable REST API) nie blokuje endpointów.</li>
                        <li>Przeanalizuj <code>.htaccess</code> oraz reguły serwera (np. mod_security).</li>
                        <li>Wyłącz wtyczki i aktywuj motyw domyślny – testuj konflikt.</li>
                    </ol>'
            ];
        }

        // 3. WP Filesystem
        if (self::is_wp_filesystem_available()) {
            $items[] = [
                'status' => 'ok',
                'title' => 'WP Filesystem – dostępny',
                'details' => '<p><strong>WP_Filesystem działa poprawnie.</strong><br> Możliwe są aktualizacje i instalacje przez interfejs WordPressa.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'WP Filesystem – niedostępny',
                'details' => '
                    <p><strong>Dlaczego WP_Filesystem jest istotny?</strong></p>
                    <ul>
                        <li>Bez niego WordPress może wymagać poświadczeń FTP.</li>
                    </ul>
                    <p><strong>Jak naprawić?</strong></p>
                    <ol>
                        <li>Upewnij się, że katalogi <code>wp-content</code> mają prawa zapisu.</li>
                        <li>Dodaj dane FTP do <code>wp-config.php</code> jeśli to wymagane:</li>
                    </ol>
                    <pre>
                        define(\'FTP_HOST\', \'ftp.twojadomena.pl\');
                        define(\'FTP_USER\', \'nazwa_uzytkownika\');
                        define(\'FTP_PASS\', \'haslo\');
                    </pre>'
            ];
        }

        // 4. Moduły PHP
        $missing_modules = self::get_missing_php_modules();
        if (empty($missing_modules)) {
            $items[] = [
                'status' => 'ok',
                'title' => 'Moduły PHP – wszystkie obecne',
                'details' => '<p><strong>Wszystkie wymagane moduły PHP są zainstalowane.</strong><br>Środowisko PHP jest zgodne z wymaganiami WordPressa.</p>'
            ];
        } else {
            $items[] = [
                'status' => 'warning',
                'title' => 'Brakuje modułów PHP: ' . implode(', ', $missing_modules),
                'details' => '
                    <p><strong>Dlaczego są potrzebne?</strong></p>
                    <ul>
                        <li>Brakujące rozszerzenia mogą powodować błędy działania wtyczek i motywów.</li>
                    </ul>
                    <p><strong>Jak naprawić?</strong></p>
                    <ol>
                        <li>Wejdź w <strong>PHP Selector</strong> (np. cPanel / Plesk).</li>
                        <li>Zaznacz brakujące moduły: ' . implode(', ', $missing_modules) . '</li>
                        <li>Zapisz zmiany i uruchom ponownie PHP jeśli to konieczne.</li>
                    </ol>'
            ];
        }

        // 5. Logi błędów PHP
        $log_info = self::get_error_log_status();
        if ($log_info['has_errors']) {
            $items[] = [
                'status' => 'warning',
                'title' => 'Error log – wykryto błędy',
                'details' => '
                    <p><strong>W logach PHP wykryto błędy.</strong></p>
                    <p><strong>Ostatnie wpisy:</strong></p>
                    <pre>' . esc_html($log_info['content']) . '</pre>
                    <p><strong>Jak rozwiązać błędy?</strong></p>
                    <ol>
                        <li>Sprawdź, która wtyczka lub motyw je generuje.</li>
                        <li>Zaktualizuj lub wyłącz problematyczne rozszerzenia.</li>
                    </ol>'
            ];
        } else {
            $items[] = [
                'status' => 'ok',
                'title' => 'Error log – brak krytycznych błędów',
                'details' => '<p><strong>Brak krytycznych błędów w logach PHP.</strong><br>Serwer nie raportuje ostatnich problemów technicznych.</p>'
            ];
        }

        return $items;
    }

    // --- POMOCNICZE ---
    public static function get_icon_for_status($status)
    {
        switch ($status) {
            case 'ok':
                return 'fas fa-check-circle';
            case 'warning':
                return 'fas fa-exclamation-triangle';
            case 'danger':
                return 'fas fa-times-circle';
            default:
                return 'fas fa-info-circle';
        }
    }

    // --- POMOCNICZE - BEZPIECZEŃSTWO ---
    private static function has_default_prefix() {
        global $wpdb;
        return $wpdb->prefix === 'wp_';
    }

    private static function check_xmlrpc_status() {
        // 1. Sprawdź czy XML-RPC jest wyłączone przez filtr
        $xmlrpc_disabled_by_filter = !apply_filters('xmlrpc_enabled', true);
        
        // 2. Sprawdź czy endpoint XML-RPC odpowiada
        $xmlrpc_url = site_url('/xmlrpc.php');
        $xmlrpc_responds = false;
        
        try {
            $response = wp_remote_post($xmlrpc_url, [
                'timeout' => 4,
                'body' => '<?xml version="1.0"?><methodCall><methodName>demo.sayHello</methodName></methodCall>',
                'headers' => [
                    'Content-Type' => 'text/xml',
                    'User-Agent' => 'WP-TECH-Security-Check'
                ]
            ]);

            $xmlrpc_responds = !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
        } catch (Exception $e) {
            // BEZPIECZEŃSTWO: Jeśli nie można sprawdzić, przyjmij że odpowiada
            $xmlrpc_responds = true;
        }
        
        // 3. Sprawdź czy .htaccess blokuje XML-RPC
        $htaccess_blocks_xmlrpc = self::htaccess_blocks_xmlrpc();

        return [
            'enabled' => !$xmlrpc_disabled_by_filter && $xmlrpc_responds && !$htaccess_blocks_xmlrpc,
            'disabled_by_filter' => $xmlrpc_disabled_by_filter,
            'blocked_by_htaccess' => $htaccess_blocks_xmlrpc,
            'responds' => $xmlrpc_responds
        ];
    }

    private static function htaccess_blocks_xmlrpc() {
        $htaccess_file = ABSPATH . '.htaccess';
        
        if (!file_exists($htaccess_file) || !is_readable($htaccess_file)) {
            return false;
        }

        try {
            $content = file_get_contents($htaccess_file);
            
            // BEZPIECZEŃSTWO: Sprawdź różne wzorce blokowania XML-RPC
            $patterns = [
                '/xmlrpc\.php/i',
                '/Files.*xmlrpc/i',
                '/FilesMatch.*xmlrpc/i',
                '/deny.*xmlrpc/i',
                '/block.*xmlrpc/i'
            ];

            foreach ($patterns as $pattern) {
                if (preg_match($pattern, $content)) {
                    return true;
                }
            }
        } catch (Exception $e) {
            // BEZPIECZEŃSTWO: Jeśli nie można odczytać, przyjmij że nie blokuje
            return false;
        }

        return false;
    }

    private static function get_problematic_plugins() {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins = get_plugins();
        $active = get_option('active_plugins', []);
        $problematic = [];

        foreach ($plugins as $file => $plugin) {
            $path = WP_PLUGIN_DIR . '/' . dirname($file);
            $modified = file_exists($path) ? filemtime($path) : false;
            $inactive = !in_array($file, $active, true);
            $no_version = empty($plugin['Version']);

            if ($no_version) {
                $problematic[] = esc_html($plugin['Name']) . ' – brak wersji';
            }

            if ($inactive && $modified && $modified < strtotime('-1 year')) {
                $problematic[] = esc_html($plugin['Name']) . ' – nieaktywna od ponad roku';
            }
        }

        return $problematic;
    }

    private static function get_insecure_permissions() {
        $bad = [];

        $files_to_check = [ABSPATH . 'wp-config.php'];
        foreach ($files_to_check as $file) {
            if (file_exists($file)) {
                $perm = substr(sprintf('%o', fileperms($file)), -4);
                if ((int)$perm > 644) {
                    $bad[] = str_replace(ABSPATH, '', $file) . " – uprawnienia: {$perm}";
                }
            }
        }

        $paths = [WP_CONTENT_DIR . '/plugins', WP_CONTENT_DIR . '/themes'];
        foreach ($paths as $path) {
            if (is_dir($path)) {
                $iter = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($path, FilesystemIterator::SKIP_DOTS));
                foreach ($iter as $file) {
                    $perm = intval(substr(sprintf('%o', $file->getPerms()), -4));
                    $rel = str_replace(ABSPATH, '', $file->getPathname());
                    if ($file->isDir() && $perm > 755) {
                        $bad[] = "Katalog {$rel} – {$perm}";
                    }
                    if ($file->isFile() && $perm > 644) {
                        $bad[] = "Plik {$rel} – {$perm}";
                    }
                }
            }
        }

        return $bad;
    }

    private static function admin_user_exists() {
        return username_exists('admin');
    }

    private static function is_debug_risky() {
        return defined('WP_DEBUG') && WP_DEBUG === true &&
            (!defined('WP_DEBUG_DISPLAY') || WP_DEBUG_DISPLAY !== false);
    }

    private static function get_accessible_php_files_in_uploads() {
        $found = [];
        $base_dir = WP_CONTENT_DIR . '/uploads';
        $base_url = content_url('/uploads');

        if (!is_dir($base_dir)) return $found;

        try {
            // BEZPIECZEŃSTWO: Limit czasu i liczby plików
            $count = 0;
            $max_files = 100; // Limit sprawdzanych plików
            
            $iter = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($base_dir, FilesystemIterator::SKIP_DOTS));
            
            foreach ($iter as $file) {
                // BEZPIECZEŃSTWO: Prevent timeout na dużych stronach
                if (++$count > $max_files) {
                    break;
                }
                
                if ($file->isFile() && strtolower($file->getExtension()) === 'php') {
                    $rel_path = str_replace($base_dir, '', $file->getPathname());
                    $url = $base_url . str_replace(DIRECTORY_SEPARATOR, '/', $rel_path);
                    
                    // BEZPIECZEŃSTWO: Krótszy timeout i User-Agent
                    $response = wp_remote_get($url, [
                        'timeout' => 3,
                        'headers' => ['User-Agent' => 'WP-TECH-Security-Check']
                    ]);
                    
                    if (!is_wp_error($response)) {
                        $code = wp_remote_retrieve_response_code($response);
                        if ($code >= 200 && $code < 300) {
                            $found[] = esc_url($url);
                        }
                    }
                }
            }
        } catch (Exception $e) {
            // BEZPIECZEŃSTWO: Nie ujawniaj błędów systemu
            return [];
        }

        return $found;
    }

    private static function get_ssl_status() {
        $is_https = is_ssl();
        $url_https = stripos(get_site_url(), 'https://') === 0;

        return [
            'is_https' => $is_https,
            'url_forced_https' => $url_https
        ];
    }

    private static function check_directory_browsing()
    {
        $accessible_dirs = [];
        
        // Testuj popularne katalogi WordPress
        $dirs_to_test = [
            '/wp-content/',
            '/wp-includes/',
            '/wp-content/themes/',
            '/wp-content/plugins/',
            '/wp-content/uploads/'
        ];

        foreach ($dirs_to_test as $dir) {
            $url = home_url($dir);
            
            try {
                $response = wp_remote_get($url, [
                    'timeout' => 4,
                    'headers' => [
                        'User-Agent' => 'WP-TECH-Security-Check',
                        'Accept' => 'text/html,application/xhtml+xml'
                    ]
                ]);

                if (!is_wp_error($response)) {
                    $body = wp_remote_retrieve_body($response);
                    $code = wp_remote_retrieve_response_code($response);
                    
                    // BEZPIECZEŃSTWO: Sprawdź tylko podstawowe wzorce directory listing
                    if ($code === 200 && !empty($body) && (
                        stripos($body, 'Index of') !== false ||
                        stripos($body, 'Directory Listing') !== false ||
                        (stripos($body, '<title>') !== false && stripos($body, $dir) !== false) ||
                        (stripos($body, '<a href=') !== false && stripos($body, 'Parent Directory') !== false)
                    )) {
                        $accessible_dirs[] = esc_url($url);
                    }
                }
            } catch (Exception $e) {
                // BEZPIECZEŃSTWO: Kontynuuj sprawdzanie pozostałych katalogów
                continue;
            }
        }

        return [
            'enabled' => !empty($accessible_dirs),
            'accessible_dirs' => $accessible_dirs
        ];
    }

    private static function is_file_editing_enabled()
    {
        // Sprawdź czy DISALLOW_FILE_EDIT jest ustawione na true
        if (defined('DISALLOW_FILE_EDIT') && DISALLOW_FILE_EDIT === true) {
            return false;
        }

        // Domyślnie WordPress pozwala na edycję plików
        return true;
    }

    // --- POMOCNICZE WYDAJNOŚĆ
    private static function get_page_load_time_frontend()
    {
        $url = home_url('/');
        $start = microtime(true);

        $response = wp_remote_get($url, [
            'timeout' => 8,
            'headers' => [
                'Cache-Control' => 'no-cache',
                'User-Agent' => 'WP-TECH-Performance-Check'
            ],
        ]);

        $ttfb = null;
        if (!is_wp_error($response) && isset($response['http_response'])) {
            $ttfb = $response['http_response']->get_response_object()->header_size
                ? intval((microtime(true) - $start) * 1000)
                : null;
        }

        return [
            'ttfb_ms' => $ttfb
        ];
    }

    private static function get_page_size()
    {
        $url = home_url('/');
        $response = wp_remote_get($url, [
            'timeout' => 8,
            'headers' => ['User-Agent' => 'WP-TECH-Performance-Check']
        ]);

        if (is_wp_error($response)) {
            return null;
        }

        $body = wp_remote_retrieve_body($response);
        $size_bytes = strlen($body);
        return round($size_bytes / 1024 / 1024, 2); // MB
    }

    private static function is_object_cache_properly_configured()
    {
        if ( ! file_exists( WP_CONTENT_DIR . '/object-cache.php' ) ) {
            return false;
        }

        if ( ! function_exists('wp_cache_get') ) {
            return false;
        }

        $test_key = 'wptech_object_cache_test_' . wp_rand();
        $test_value = 'ok_' . time();

        wp_cache_set($test_key, $test_value, '', 30);
        $fetched = wp_cache_get($test_key);

        return $fetched === $test_value;
    }

    private static function is_html_cache_enabled()
    {
        // 1. Sprawdź popularne wtyczki cache
        $active_plugins = apply_filters('active_plugins', get_option('active_plugins', []));
        $has_plugin = false;

        $cache_plugins = [
            'litespeed-cache/litespeed-cache.php',
            'wp-rocket/wp-rocket.php',
            'w3-total-cache/w3-total-cache.php',
            'autoptimize/autoptimize.php',
            'sg-cachepress/sg-cachepress.php',
        ];

        foreach ($cache_plugins as $plugin) {
            if (in_array($plugin, $active_plugins, true)) {
                $has_plugin = true;
                break;
            }
        }

        // 2. Sprawdź nagłówki z bufora
        $response = wp_remote_get(home_url('/'), [
            'timeout' => 6,
            'headers' => ['User-Agent' => 'WP-TECH-Cache-Check']
        ]);

        if (is_wp_error($response)) {
            return $has_plugin; // fallback: wykryto wtyczkę, nie można ocenić nagłówków
        }

        $headers = wp_remote_retrieve_headers($response);
        $cache_headers = ['x-cache', 'x-litespeed-cache', 'x-powered-by', 'cf-cache-status', 'x-cache-status'];

        foreach ($cache_headers as $key) {
            if (!empty($headers[$key])) {
                return true;
            }
        }

        // 3. Jeśli żadnych śladów, ale wtyczka istnieje – przyjmij jako potencjalnie aktywną
        return $has_plugin;
    }

    private static function detect_cdn()
    {
        $response = wp_remote_get(home_url('/'), [
            'timeout' => 6,
            'headers' => ['User-Agent' => 'WP-TECH-CDN-Check']
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $headers = wp_remote_retrieve_headers($response);
        $indicators = [
            'cf-cache-status' => 'Cloudflare',
            'x-cdn' => 'Generic CDN',
            'x-proxy-cache' => 'Nginx Proxy Cache',
            'x-cache' => 'Amazon CloudFront',
            'server' => [
                'cloudflare' => 'Cloudflare',
                'bunnycdn' => 'BunnyCDN',
                'akamai' => 'Akamai',
            ],
        ];

        foreach ($indicators as $key => $value) {
            if (is_array($value) && isset($headers[$key])) {
                foreach ($value as $sig => $cdn_name) {
                    if (stripos($headers[$key], $sig) !== false) {
                        return $cdn_name;
                    }
                }
            } elseif (isset($headers[$key])) {
                return $value;
            }
        }

        return false;
    }

    private static function get_php_memory_limit()
    {
        $raw = ini_get('memory_limit');
        if (is_numeric($raw)) return (int) $raw;

        $unit = strtolower(substr($raw, -1));
        $bytes = (int) $raw;

        switch ($unit) {
            case 'g': $bytes *= 1024;
            case 'm': return $bytes;
            case 'k': return (int)($bytes / 1024);
            default: return 0;
        }
    }

    private static function get_max_execution_time()
    {
        return (int) ini_get('max_execution_time');
    }

    // --- POMOCNICZE AKTUALNOŚĆ
    private static function get_php_version() {
        return PHP_VERSION;
    }

    private static function get_latest_wp_version() {
        $transient = get_transient('wptech_latest_wp_version');
        if ($transient) {
            return $transient;
        }

        try {
            $response = wp_remote_get('https://api.wordpress.org/core/version-check/1.7/', [
                'timeout' => 5,
                'headers' => ['User-Agent' => 'WP-TECH-Update-Check']
            ]);
            
            if (is_wp_error($response)) {
                return get_bloginfo('version');
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);
            if (!empty($body['offers'][0]['version'])) {
                set_transient('wptech_latest_wp_version', $body['offers'][0]['version'], 12 * HOUR_IN_SECONDS);
                return $body['offers'][0]['version'];
            }
        } catch (Exception $e) {
            // BEZPIECZEŃSTWO: W przypadku błędu zwróć obecną wersję
            return get_bloginfo('version');
        }

        return get_bloginfo('version');
    }

    private static function get_database_engine_type() {
        global $wpdb;
        $version = $wpdb->get_var("SELECT VERSION()");

        if (stripos($version, 'MariaDB') !== false) {
            return 'MariaDB';
        }
        return 'MySQL';
    }

    private static function get_latest_mysql_version() {
        return '8.0.36';
    }

    private static function get_latest_mariadb_version() {
        return '11.3.2';
    }

    private static function get_outdated_plugins_list() {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        $all_plugins = get_plugins();
        $updates = get_site_transient('update_plugins');
        $outdated = [];

        foreach ($all_plugins as $path => $plugin) {
            $slug = dirname($path);
            if (isset($updates->response[$path])) {
                $outdated[] = $plugin['Name'] . ' (' . $plugin['Version'] . ' → ' . $updates->response[$path]->new_version . ')';
            }
        }

        return $outdated;
    }

    private static function is_auto_update_enabled() {
        if (defined('WP_AUTO_UPDATE_CORE') && WP_AUTO_UPDATE_CORE === true) {
            return true;
        }

        // Fallback: sprawdzenie przez filtr
        return apply_filters('allow_major_auto_core_updates', false) === true;
    }

    // --- POMOCNICZE STABILNOŚĆ
    private static function has_cron_jobs()
    {
        $crons = _get_cron_array();
        return is_array($crons) && count($crons) > 0;
    }

    private static function is_rest_api_available()
    {
        $response = wp_remote_get(rest_url('wp/v2'));
        return !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
    }

    private static function is_wp_filesystem_available()
    {
        if (!function_exists('WP_Filesystem')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        global $wp_filesystem;
        WP_Filesystem();

        return is_object($wp_filesystem) && $wp_filesystem->method !== 'ftpext';
    }

    private static function get_missing_php_modules()
    {
        $required = ['curl', 'mbstring', 'json', 'openssl', 'dom', 'zip', 'gd'];
        return array_values(array_filter($required, fn($ext) => !extension_loaded($ext)));
    }

    private static function get_error_log_status()
    {
        $log_file = ini_get('error_log');

        if (!$log_file || !file_exists($log_file) || !is_readable($log_file)) {
            return ['has_errors' => false, 'content' => 'Brak dostępu do pliku logów.'];
        }

        $content = file_get_contents($log_file);
        $lines = explode("\n", trim($content));
        $recent = array_slice($lines, -10); // ostatnie 10 wpisów

        $has_errors = !empty(array_filter($recent, fn($line) => stripos($line, 'error') !== false));

        return [
            'has_errors' => $has_errors,
            'content' => implode("\n", $recent)
        ];
    }
}